<?php

namespace Teamnet\FaxApi\Soap\Client;

use ErrorException;
use SoapClient;
use SoapFault;
use Teamnet\FaxApi\Soap\Types\APIJob;
use Teamnet\FaxApi\Soap\Types\APIJobProfile;
use Teamnet\FaxApi\Soap\Types\AuthenticationException;
use Teamnet\FaxApi\Soap\Types\GetJob;
use Teamnet\FaxApi\Soap\Types\GetJobCosts;
use Teamnet\FaxApi\Soap\Types\GetJobCostsResponse;
use Teamnet\FaxApi\Soap\Types\GetJobMembers;
use Teamnet\FaxApi\Soap\Types\GetJobMembersResponse;
use Teamnet\FaxApi\Soap\Types\GetJobProfile;
use Teamnet\FaxApi\Soap\Types\GetJobProfileResponse;
use Teamnet\FaxApi\Soap\Types\GetJobResponse;
use Teamnet\FaxApi\Soap\Types\GetList;
use Teamnet\FaxApi\Soap\Types\GetMembers;
use Teamnet\FaxApi\Soap\Types\GetMembersResponse;
use Teamnet\FaxApi\Soap\Types\GetTransportStatus;
use Teamnet\FaxApi\Soap\Types\GetTransportStatusResponse;
use Teamnet\FaxApi\Soap\Types\JobList;
use Teamnet\FaxApi\Soap\Types\JobNotFoundException;
use Teamnet\FaxApi\Soap\Types\JobTypes;
use Teamnet\FaxApi\Soap\Types\MaintenanceException;
use Teamnet\FaxApi\Soap\Types\ParameterException;
use Teamnet\FaxApi\Soap\Types\Period;
use Teamnet\FaxApi\Soap\Types\ServiceException;
use Teamnet\FaxApi\Soap\Types\SetTransportStatus;

/**
 * Client class to obtain or alter faxjobs using the Teamnet FaxAPI webservice
 * @author tingelhoff@teamnet.de
 * @copyright Teamnet GmbH
 * @version 1.0
 */
class Job {
	/**
	 * @var SoapClient
	 */
	private $oSOAP;
	private $authKey;
	
	
	
	/**
	 * @throws ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl, $authKey ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			$this->authKey = $authKey;
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => JobTypes::getClassMap() ) );
		}
		catch ( SoapFault $exception ) {
			throw new ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Create a period object
	 *  
	 * @param string $sFrom Begin of the period formatted as ISO 8601 date
	 * @param string $sUntil End of the period formatted as ISO 8601 date
	 */
	public static function createPeriod( $sFrom, $sUntil ) {
		$oPeriod = new Period();
		$oPeriod->from = $sFrom;
		$oPeriod->until = $sUntil;
		return $oPeriod;
	}
	
	
	
	/**
	 * Gets a list of members associated to a specific job
	 * @param int $iJobID
	 * @return array of APIJobMemberList objects
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getJobMembers($iJobID) {
		$oReq = new GetJobMembers();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetJobMembersResponse $oResponse */
			$oResponse = $this->oSOAP->getJobMembers($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJobMemberList->apiJobMemberList;
	}
	
	
	
	/**
	 * Gets the costs of a specific job
	 * @param int $iJobID
	 * @return float|null
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getJobCosts($iJobID) {
		$oReq = new GetJobCosts();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetJobCostsResponse $oResponse */
			$oResponse = $this->oSOAP->getJobCosts($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->costs;
	}
	
	
	
	/**
	 * Gets the current status of a specific job
	 * @param int $iJobID JobID
	 * @return APIJob
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getJob($iJobID) {
		$oReq = new GetJob();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetJobResponse $oResponse */
			$oResponse = $this->oSOAP->getJob($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJob;
	}
	
	
	
	/**
	 *
	 * @param int $iJobID JobID
	 * @return APIJobProfile
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getJobProfile($iJobID) {
		$oReq = new GetJobProfile();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetJobProfileResponse $oResponse */
			$oResponse = $this->oSOAP->getJobProfile($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->apiJobProfile;
	}
	
	
	
	/**
	 * Returns a list of APIJob objects matching the delivered search parameters
	 * @param string|null	$sTitle
	 * @param string|null	$sStatus
	 * @param Period|null	$oPeriod
	 * @return array						APIJob objects
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getList($sTitle = null, $sStatus = null, Period $oPeriod = null) {
		$oReq = new GetList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sTitle;
		$oReq->status = $sStatus;
		$oReq->period = $oPeriod;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var JobList $jobList */
			$jobList = $this->oSOAP->getList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $jobList->jobList->jobList;
	}
	
	
	
	/**
	 * Returns a list of APIJobMember objects matching the delivered search parameters
	 * @param string|null	$sTitle
	 * @param string|null	$sStatus
	 * @param Period|null	$oPeriod
	 * @return array						APIJobMember objects
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getMembers($sTitle = null, $sStatus = null, Period $oPeriod = null) {
		$oReq = new GetMembers();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sTitle;
		$oReq->status = $sStatus;
		$oReq->period = $oPeriod;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetMembersResponse $jobList */
			$jobList = $this->oSOAP->getMembers($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $jobList->apiJobMemberList->apiJobMemberList;
	}
	
	
	
	/**
	 * Gets the current transport state of a specific job
	 * @param int $iJobID
	 * @return string	the current transport status
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function getTransportStatus($iJobID) {
		$oReq = new GetTransportStatus();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var GetTransportStatusResponse $oResponse */
			$oResponse = $this->oSOAP->getTransportStatus($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->transportStatus;
	}
	
	
	
	/**
	 * Alters the transport status of a specific job
	 * @param int		$iJobID
	 * @param string	$sTransportStatus	the transport status to set
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws JobNotFoundException			if a job with the given ID was not found
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function setTransportStatus($iJobID, $sTransportStatus) {
		$oReq = new SetTransportStatus();
		$oReq->authKey = $this->authKey;
		$oReq->jobId = $iJobID;
		$oReq->transportStatus = $sTransportStatus;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->oSOAP->setTransportStatus($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
	}
	
	
	
	/**
	 * @param SoapFault $oSoapFault
	 * @return mixed
	 * @internal
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = '\\Teamnet\\FaxApi\\Soap\\Types\\'.array_pop( $keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
	
	
	
	/**
	 * @return SoapClient
	 */
	public function getSoapClient(): SoapClient {
		return $this->oSOAP;
	}
}
