<?php

namespace Teamnet\FaxApi\Soap\Client;

use ErrorException;
use SoapClient;
use SoapFault;
use Teamnet\FaxApi\Soap\Types\AuthenticationException;
use Teamnet\FaxApi\Soap\Types\FaxRecipientList;
use Teamnet\FaxApi\Soap\Types\JobOptions;
use Teamnet\FaxApi\Soap\Types\LimitExceededException;
use Teamnet\FaxApi\Soap\Types\MaintenanceException;
use Teamnet\FaxApi\Soap\Types\ParameterException;
use Teamnet\FaxApi\Soap\Types\SendFaxResponse;
use Teamnet\FaxApi\Soap\Types\SendFaxToDistributionList;
use Teamnet\FaxApi\Soap\Types\SendFaxToFaxnumber;
use Teamnet\FaxApi\Soap\Types\SendFaxToFaxRecipientList;
use Teamnet\FaxApi\Soap\Types\SendfaxTypes;
use Teamnet\FaxApi\Soap\Types\ServiceException;

/**
 * Client class to send faxes using the Teamnet FaxAPI webservice
 * @author tingelhoff@teamnet.de
 * @copyright Teamnet GmbH
 * @version 1.0
 */
class SendFax {
	/**
	 * @var SoapClient
	 */
	private $oSOAP;
	private $authKey;
	
	
	
	/**
	 * @throws ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl, $authKey ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			
			$this->authKey = $authKey;
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => SendfaxTypes::getClassMap() ) );
		} catch ( SoapFault $exception ) {
			throw new ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Sends a fax to one specified recipient
	 * @param string			$sJobTitle
	 * @param string			$sContentType
	 * @param string			$sContent
	 * @param string			$sFaxnumber		Faxnummer
	 * @param JobOptions|null	$oJobOptions
	 * @return int JobID
	 * @throws ServiceException				if an internal error in the fax service occurred
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault					if another SOAP related error occurred
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function sendFaxToFaxnumber($sJobTitle, $sContentType, $sContent,
									   $sFaxnumber, JobOptions $oJobOptions = null)
	{
		$oReq = new SendFaxToFaxnumber();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $sContentType;
		$oReq->content = $sContent;
		$oReq->faxnumber = $sFaxnumber;
		$oReq->jobOptions = $oJobOptions;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToFaxnumber($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * Sends a fax to a list of recipients
	 * @param string								$sJobTitle
	 * @param string|int							$mContentType
	 * @param string								$sContent
	 * @param RecipientList							$aRecipientList	Array von FaxRecipient objekten
	 * @param JobOptions|null						$oJobOptions
	 * @return int JobID
	 * @throws ServiceException				if an internal error in the fax service occured
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException		if the given AuthKey is invalid
	 * @throws LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault					if another SOAP related error occured
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function sendFaxToRecipientList($sJobTitle, $mContentType, $sContent,
										  RecipientList $oRecipientList,
										   JobOptions $oJobOptions = null)
	{
		$oReq = new SendFaxToFaxRecipientList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $mContentType;
		$oReq->content = $sContent;
		$oReq->faxRecipientList = $oList = new FaxRecipientList();
		$oReq->jobOptions = $oJobOptions;
		
		$oList->faxRecipient = $oRecipientList->getRecipients();
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToFaxRecipientList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * Sends a fax to a distribution list
	 * @param string			$sJobTitle
	 * @param string|int		$mContentType
	 * @param string			$sContent
	 * @param int				$iDistributionListId	ID of distribution list
	 * @param JobOptions|null	$oJobOptions
	 * @return int JobID
	 * @throws ServiceException				if an internal error in the fax service occured
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws AuthenticationException	if the given AuthKey is invalid
	 * @throws LimitExceededException		if you try to send to many jobs in a short time
	 * @throws SoapFault					if another SOAP related error occured
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function sendFaxToDistributionList($sJobTitle, $mContentType, $sContent,
											  $iDistributionListId, JobOptions $oJobOptions = null)
	{
		$oReq = new SendFaxToDistributionList();
		$oReq->authKey = $this->authKey;
		$oReq->jobTitle = $sJobTitle;
		$oReq->contentType = $mContentType;
		$oReq->content = $sContent;
		$oReq->distributionListId = $iDistributionListId;
		$oReq->jobOptions = $oJobOptions;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			/** @var SendFaxResponse $oResponse */
			$oResponse = $this->oSOAP->sendFaxToDistributionList($oReq);
		} catch (SoapFault $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
		
		return $oResponse->jobId;
	}
	
	
	
	/**
	 * @param SoapFault $oSoapFault
	 * @return mixed
	 * @internal
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = '\\Teamnet\\FaxApi\\Soap\\Types\\'.array_pop( $keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
}
