<?php

namespace Teamnet\FaxApi\Soap\Client;

use ErrorException;
use Exception;
use SoapClient;
use SoapFault;
use Teamnet\FaxApi\Soap\Types\AuthenticationException;
use Teamnet\FaxApi\Soap\Types\CheckAuthenticationRequest;
use Teamnet\FaxApi\Soap\Types\CheckVersionRequest;
use Teamnet\FaxApi\Soap\Types\LimitExceededException;
use Teamnet\FaxApi\Soap\Types\MaintenanceException;
use Teamnet\FaxApi\Soap\Types\ParameterException;
use Teamnet\FaxApi\Soap\Types\ServiceException;



/**
 * Client class for QuickStart related tasks, using the Teamnet FaxAPI webservice
 * @author tingelhoff@teamnet.de
 * @copyright Teamnet GmbH
 * @version 1.0
 */
class Service {
	/**
	 * @var SoapClient
	 */
	public $oSOAP;
	
	
	
	/**
	 * @throws ServiceException
	 * @throws ErrorException
	 */
	public function __construct($wsdl ) {
		try {
			if ( !isset( $wsdl ) ) {
				throw new ErrorException( 'WSDL isn\'t set.' );
			}
			$this->oSOAP = new SoapClient( (string) $wsdl, array( 'classmap' => \Teamnet\FaxApi\Soap\Handler\Sendfax::getClassMap() ));
		}
		catch ( SoapFault $exception ) {
			throw new ServiceException( $exception );
		}
	}
	
	
	
	/**
	 * Checks whether the given AuthKey is valid or not
	 * @param string $authkey
	 * @return bool
	 * @throws ServiceException				if an internal error in the fax service occured
	 * @throws MaintenanceException			if the fax service is currently down for maintenance
	 * @throws ParameterException			if parameters are invalid
	 * @throws SoapFault					if another SOAP related error occured
	 * @noinspection PhpDocRedundantThrowsInspection
	 */
	public function checkAuthentication($authkey) {
		$oCheckAuthRequest = new CheckAuthenticationRequest();
		$oCheckAuthRequest->authKey = $authkey;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->oSOAP->checkAuthentication($oCheckAuthRequest);
		} catch (SoapFault $e) {
			if (! isset($e->detail)) {
				throw $e;
			}
			
			$oException = $this->_convertSoapFault($e);
			
			if ($oException instanceof AuthenticationException)
				return false;
			
			throw $oException;
		}
		
		return true;
	}
	
	
	
	/**
	 * Used by QuickStart to check for a new version
	 * @param string $sVersion
	 * @param string|null $sSysInfo
	 * @throws ServiceException                if an internal error in the fax service occured
	 * @throws MaintenanceException            if the fax service is currently down for maintenance
	 * @throws ParameterException            if parameters are invalid
	 * @throws SoapFault                    if another SOAP related error occured
	 * @throws Exception
	 */
	public function checkVersion($sVersion, $sSysInfo = null) {
		$req = new CheckVersionRequest();
		$req->system = 'PHP';
		$req->qsVersion = $sVersion;
		$req->systemInfo = $sSysInfo;
		
		try {
			/** @noinspection PhpUndefinedMethodInspection */
			return $this->oSOAP->checkVersion($req);
		} catch (Exception $e) {
			throw isset($e->detail) ? $this->_convertSoapFault($e) : $e;
		}
	}
	
	
	
	/**
	 * @internal
	 * @param SoapFault $oSoapFault
	 */
	private function _convertSoapFault(SoapFault $oSoapFault) {
		$m_Detail = $oSoapFault->detail;
		$a_Detail = (array)$m_Detail;
		
		$keys = array_keys($a_Detail);
		$s_ExceptionType = '\\Teamnet\\FaxApi\\Soap\\Types\\'.array_pop( $keys );
		
		$o_Exception = new $s_ExceptionType();
		$o_Exception->parseSOAPFault($oSoapFault);
		
		return $o_Exception;
	}
}
